/*
 * Decompiled with CFR 0.152.
 */
package org.traccar.database;

import jakarta.annotation.Nullable;
import jakarta.inject.Inject;
import jakarta.inject.Singleton;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.traccar.config.Config;
import org.traccar.config.Keys;
import org.traccar.forward.EventData;
import org.traccar.forward.EventForwarder;
import org.traccar.geocoder.Geocoder;
import org.traccar.helper.DateUtil;
import org.traccar.model.Calendar;
import org.traccar.model.Device;
import org.traccar.model.Event;
import org.traccar.model.Geofence;
import org.traccar.model.Maintenance;
import org.traccar.model.Notification;
import org.traccar.model.Position;
import org.traccar.model.User;
import org.traccar.notification.MessageException;
import org.traccar.notification.NotificatorManager;
import org.traccar.session.cache.CacheManager;
import org.traccar.storage.Storage;
import org.traccar.storage.StorageException;
import org.traccar.storage.query.Columns;
import org.traccar.storage.query.Request;

@Singleton
public class NotificationManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(NotificationManager.class);
    private final Storage storage;
    private final CacheManager cacheManager;
    private final EventForwarder eventForwarder;
    private final NotificatorManager notificatorManager;
    private final Geocoder geocoder;
    private final boolean geocodeOnRequest;
    private final long timeThreshold;
    private final Set<Long> blockedUsers = new HashSet<Long>();

    @Inject
    public NotificationManager(Config config, Storage storage, CacheManager cacheManager, @Nullable EventForwarder eventForwarder, NotificatorManager notificatorManager, @Nullable Geocoder geocoder) {
        this.storage = storage;
        this.cacheManager = cacheManager;
        this.eventForwarder = eventForwarder;
        this.notificatorManager = notificatorManager;
        this.geocoder = geocoder;
        this.geocodeOnRequest = config.getBoolean(Keys.GEOCODER_ON_REQUEST);
        this.timeThreshold = config.getLong(Keys.NOTIFICATOR_TIME_THRESHOLD);
        String blockedUsersString = config.getString(Keys.NOTIFICATION_BLOCK_USERS);
        if (blockedUsersString != null) {
            for (String userIdString : blockedUsersString.split(",")) {
                this.blockedUsers.add(Long.parseLong(userIdString));
            }
        }
    }

    private void updateEvent(Event event, Position position) {
        try {
            event.setId(this.storage.addObject(event, new Request(new Columns.Exclude("id"))));
        }
        catch (StorageException error) {
            LOGGER.warn("Event save error", (Throwable)error);
        }
        this.forwardEvent(event, position);
        if (System.currentTimeMillis() - event.getEventTime().getTime() > this.timeThreshold) {
            LOGGER.info("Skipping notifications for old event");
            return;
        }
        List<Notification> notifications = this.cacheManager.getDeviceNotifications(event.getDeviceId()).stream().filter(notification -> notification.getType().equals(event.getType())).filter(notification -> {
            if (event.getType().equals("alarm")) {
                String alarmsAttribute = notification.getString("alarms");
                if (alarmsAttribute != null) {
                    return Arrays.asList(alarmsAttribute.split(",")).contains(event.getString("alarm"));
                }
                return false;
            }
            return true;
        }).filter(notification -> {
            String geofenceIdsString = notification.getString("geofenceIds");
            if (geofenceIdsString != null) {
                return Arrays.stream(geofenceIdsString.split(",")).map(Long::parseLong).anyMatch(geofenceId -> geofenceId.longValue() == event.getGeofenceId());
            }
            return true;
        }).filter(notification -> {
            long calendarId = notification.getCalendarId();
            Calendar calendar = calendarId != 0L ? this.cacheManager.getObject(Calendar.class, calendarId) : null;
            return calendar == null || calendar.checkMoment(event.getEventTime());
        }).toList();
        Device device = this.cacheManager.getObject(Device.class, event.getDeviceId());
        LOGGER.info("Event id: {}, time: {}, type: {}, notifications: {}", new Object[]{device.getUniqueId(), DateUtil.formatDate(event.getEventTime(), false), event.getType(), notifications.size()});
        if (!notifications.isEmpty()) {
            if (position != null && position.getAddress() == null && this.geocodeOnRequest && this.geocoder != null) {
                position.setAddress(this.geocoder.getAddress(position.getLatitude(), position.getLongitude(), null));
            }
            notifications.forEach(notification -> this.cacheManager.getNotificationUsers(notification.getId(), event.getDeviceId()).forEach(user -> {
                if (this.blockedUsers.contains(user.getId())) {
                    LOGGER.info("User {} notification blocked", (Object)user.getId());
                    return;
                }
                for (String notificator : notification.getNotificatorsTypes()) {
                    try {
                        this.notificatorManager.getNotificator(notificator).send((Notification)notification, (User)user, event, position);
                    }
                    catch (MessageException exception) {
                        LOGGER.warn("Notification failed", (Throwable)exception);
                    }
                }
            }));
        }
    }

    private void forwardEvent(Event event, Position position) {
        if (this.eventForwarder != null) {
            EventData eventData = new EventData();
            eventData.setEvent(event);
            eventData.setPosition(position);
            eventData.setDevice(this.cacheManager.getObject(Device.class, event.getDeviceId()));
            if (event.getGeofenceId() != 0L) {
                eventData.setGeofence(this.cacheManager.getObject(Geofence.class, event.getGeofenceId()));
            }
            if (event.getMaintenanceId() != 0L) {
                eventData.setMaintenance(this.cacheManager.getObject(Maintenance.class, event.getMaintenanceId()));
            }
            this.eventForwarder.forward(eventData, (success, throwable) -> {
                if (!success) {
                    LOGGER.warn("Event forwarding failed", throwable);
                }
            });
        }
    }

    public void updateEvents(Map<Event, Position> events) {
        for (Map.Entry<Event, Position> entry : events.entrySet()) {
            Event event = entry.getKey();
            Position position = entry.getValue();
            Object key = new Object();
            try {
                this.cacheManager.addDevice(event.getDeviceId(), key);
                this.updateEvent(event, position);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
            finally {
                this.cacheManager.removeDevice(event.getDeviceId(), key);
            }
        }
    }
}

